<?php
// api/register.php

// Headers para permitir el acceso desde cualquier origen (CORS) y definir el tipo de contenido
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST");

// Incluir los archivos de configuración y utilidades
require_once '../config/db.php';
require_once '../includes/mailer.php';

// Obtener los datos enviados en el cuerpo de la solicitud POST (formato JSON)
$data = json_decode(file_get_contents("php://input"));

// --- Validaciones básicas ---
if (
    empty($data->nombre) ||
    empty($data->email) ||
    empty($data->contrasena) ||
    empty($data->celular)
) {
    http_response_code(400); // Bad Request
    echo json_encode(["mensaje" => "Datos incompletos. Nombre, email, contraseña y celular son obligatorios."]);
    exit();
}

if (!filter_var($data->email, FILTER_VALIDATE_EMAIL)) {
    http_response_code(400);
    echo json_encode(["mensaje" => "Formato de email inválido."]);
    exit();
}

if (!preg_match('/^\+?[0-9]{10,15}$/', $data->celular)) {
    http_response_code(400);
    echo json_encode(["mensaje" => "Formato de celular inválido."]);
    exit();
}

// --- Lógica de negocio ---
$nombre = $data->nombre;
$email = $data->email;
$contrasena = $data->contrasena;
$celular = $data->celular;
$instagram = !empty($data->instagram) ? $data->instagram : null;

// Verificar si el email ya existe
$stmt = $mysqli->prepare("SELECT id FROM usuarios WHERE email = ?");
$stmt->bind_param("s", $email);
$stmt->execute();
$stmt->store_result();

if ($stmt->num_rows > 0) {
    http_response_code(409); // Conflict
    echo json_encode(["mensaje" => "El email ya está registrado."]);
    $stmt->close();
    exit();
}
$stmt->close();

// Hashear la contraseña
$contrasena_hash = password_hash($contrasena, PASSWORD_BCRYPT);

// Generar token de confirmación
$token = bin2hex(random_bytes(32));
$vencimiento = date('Y-m-d H:i:s', strtotime('+24 hours'));

// Insertar el nuevo usuario en la base de datos
$sql = "INSERT INTO usuarios (nombre, email, contrasena, celular, instagram, token_confirmacion, token_vencimiento) VALUES (?, ?, ?, ?, ?, ?, ?)";
$stmt = $mysqli->prepare($sql);
$stmt->bind_param("sssssss", $nombre, $email, $contrasena_hash, $celular, $instagram, $token, $vencimiento);

if ($stmt->execute()) {
    // Enviar email de confirmación
    $asunto = "Confirma tu cuenta en Rifas Doradas";
    $link_confirmacion = "http://localhost/rifas-doradas/confirmar.php?token=" . $token;
    $cuerpo_mensaje = "<h1>¡Bienvenido a Rifas Doradas!</h1>";
    $cuerpo_mensaje .= "<p>Para activar tu cuenta, por favor haz clic en el siguiente enlace:</p>";
    $cuerpo_mensaje .= "<a href='{$link_confirmacion}'>Confirmar mi cuenta</a>";
    $cuerpo_mensaje .= "<p>El enlace es válido por 24 horas.</p>";

    if (enviarEmail($email, $nombre, $asunto, $cuerpo_mensaje)) {
        http_response_code(201); // Created
        echo json_encode(["mensaje" => "Registro exitoso. Se ha enviado un email de confirmación."]);
    } else {
        http_response_code(500); // Internal Server Error
        // En un caso real, aquí se debería registrar el error de envío de email
        echo json_encode(["mensaje" => "El usuario fue creado, pero no se pudo enviar el email de confirmación."]);
    }
} else {
    http_response_code(500);
    echo json_encode(["mensaje" => "Error al registrar el usuario.", "error" => $stmt->error]);
}

$stmt->close();
$mysqli->close();
?>